﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dms/DatabaseMigrationServiceRequest.h>
#include <aws/dms/DatabaseMigrationService_EXPORTS.h>
#include <aws/dms/model/ComputeConfig.h>
#include <aws/dms/model/MigrationTypeValue.h>

#include <utility>

namespace Aws {
namespace DatabaseMigrationService {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dms-2016-01-01/ModifyReplicationConfigMessage">AWS
 * API Reference</a></p>
 */
class ModifyReplicationConfigRequest : public DatabaseMigrationServiceRequest {
 public:
  AWS_DATABASEMIGRATIONSERVICE_API ModifyReplicationConfigRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyReplicationConfig"; }

  AWS_DATABASEMIGRATIONSERVICE_API Aws::String SerializePayload() const override;

  AWS_DATABASEMIGRATIONSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name of the replication to modify.</p>
   */
  inline const Aws::String& GetReplicationConfigArn() const { return m_replicationConfigArn; }
  inline bool ReplicationConfigArnHasBeenSet() const { return m_replicationConfigArnHasBeenSet; }
  template <typename ReplicationConfigArnT = Aws::String>
  void SetReplicationConfigArn(ReplicationConfigArnT&& value) {
    m_replicationConfigArnHasBeenSet = true;
    m_replicationConfigArn = std::forward<ReplicationConfigArnT>(value);
  }
  template <typename ReplicationConfigArnT = Aws::String>
  ModifyReplicationConfigRequest& WithReplicationConfigArn(ReplicationConfigArnT&& value) {
    SetReplicationConfigArn(std::forward<ReplicationConfigArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new replication config to apply to the replication.</p>
   */
  inline const Aws::String& GetReplicationConfigIdentifier() const { return m_replicationConfigIdentifier; }
  inline bool ReplicationConfigIdentifierHasBeenSet() const { return m_replicationConfigIdentifierHasBeenSet; }
  template <typename ReplicationConfigIdentifierT = Aws::String>
  void SetReplicationConfigIdentifier(ReplicationConfigIdentifierT&& value) {
    m_replicationConfigIdentifierHasBeenSet = true;
    m_replicationConfigIdentifier = std::forward<ReplicationConfigIdentifierT>(value);
  }
  template <typename ReplicationConfigIdentifierT = Aws::String>
  ModifyReplicationConfigRequest& WithReplicationConfigIdentifier(ReplicationConfigIdentifierT&& value) {
    SetReplicationConfigIdentifier(std::forward<ReplicationConfigIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of replication.</p>
   */
  inline MigrationTypeValue GetReplicationType() const { return m_replicationType; }
  inline bool ReplicationTypeHasBeenSet() const { return m_replicationTypeHasBeenSet; }
  inline void SetReplicationType(MigrationTypeValue value) {
    m_replicationTypeHasBeenSet = true;
    m_replicationType = value;
  }
  inline ModifyReplicationConfigRequest& WithReplicationType(MigrationTypeValue value) {
    SetReplicationType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Table mappings specified in the replication.</p>
   */
  inline const Aws::String& GetTableMappings() const { return m_tableMappings; }
  inline bool TableMappingsHasBeenSet() const { return m_tableMappingsHasBeenSet; }
  template <typename TableMappingsT = Aws::String>
  void SetTableMappings(TableMappingsT&& value) {
    m_tableMappingsHasBeenSet = true;
    m_tableMappings = std::forward<TableMappingsT>(value);
  }
  template <typename TableMappingsT = Aws::String>
  ModifyReplicationConfigRequest& WithTableMappings(TableMappingsT&& value) {
    SetTableMappings(std::forward<TableMappingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The settings for the replication.</p>
   */
  inline const Aws::String& GetReplicationSettings() const { return m_replicationSettings; }
  inline bool ReplicationSettingsHasBeenSet() const { return m_replicationSettingsHasBeenSet; }
  template <typename ReplicationSettingsT = Aws::String>
  void SetReplicationSettings(ReplicationSettingsT&& value) {
    m_replicationSettingsHasBeenSet = true;
    m_replicationSettings = std::forward<ReplicationSettingsT>(value);
  }
  template <typename ReplicationSettingsT = Aws::String>
  ModifyReplicationConfigRequest& WithReplicationSettings(ReplicationSettingsT&& value) {
    SetReplicationSettings(std::forward<ReplicationSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Additional settings for the replication.</p>
   */
  inline const Aws::String& GetSupplementalSettings() const { return m_supplementalSettings; }
  inline bool SupplementalSettingsHasBeenSet() const { return m_supplementalSettingsHasBeenSet; }
  template <typename SupplementalSettingsT = Aws::String>
  void SetSupplementalSettings(SupplementalSettingsT&& value) {
    m_supplementalSettingsHasBeenSet = true;
    m_supplementalSettings = std::forward<SupplementalSettingsT>(value);
  }
  template <typename SupplementalSettingsT = Aws::String>
  ModifyReplicationConfigRequest& WithSupplementalSettings(SupplementalSettingsT&& value) {
    SetSupplementalSettings(std::forward<SupplementalSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration parameters for provisioning an DMS Serverless replication.</p>
   */
  inline const ComputeConfig& GetComputeConfig() const { return m_computeConfig; }
  inline bool ComputeConfigHasBeenSet() const { return m_computeConfigHasBeenSet; }
  template <typename ComputeConfigT = ComputeConfig>
  void SetComputeConfig(ComputeConfigT&& value) {
    m_computeConfigHasBeenSet = true;
    m_computeConfig = std::forward<ComputeConfigT>(value);
  }
  template <typename ComputeConfigT = ComputeConfig>
  ModifyReplicationConfigRequest& WithComputeConfig(ComputeConfigT&& value) {
    SetComputeConfig(std::forward<ComputeConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the source endpoint for this DMS serverless
   * replication configuration.</p>
   */
  inline const Aws::String& GetSourceEndpointArn() const { return m_sourceEndpointArn; }
  inline bool SourceEndpointArnHasBeenSet() const { return m_sourceEndpointArnHasBeenSet; }
  template <typename SourceEndpointArnT = Aws::String>
  void SetSourceEndpointArn(SourceEndpointArnT&& value) {
    m_sourceEndpointArnHasBeenSet = true;
    m_sourceEndpointArn = std::forward<SourceEndpointArnT>(value);
  }
  template <typename SourceEndpointArnT = Aws::String>
  ModifyReplicationConfigRequest& WithSourceEndpointArn(SourceEndpointArnT&& value) {
    SetSourceEndpointArn(std::forward<SourceEndpointArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the target endpoint for this DMS serverless
   * replication configuration.</p>
   */
  inline const Aws::String& GetTargetEndpointArn() const { return m_targetEndpointArn; }
  inline bool TargetEndpointArnHasBeenSet() const { return m_targetEndpointArnHasBeenSet; }
  template <typename TargetEndpointArnT = Aws::String>
  void SetTargetEndpointArn(TargetEndpointArnT&& value) {
    m_targetEndpointArnHasBeenSet = true;
    m_targetEndpointArn = std::forward<TargetEndpointArnT>(value);
  }
  template <typename TargetEndpointArnT = Aws::String>
  ModifyReplicationConfigRequest& WithTargetEndpointArn(TargetEndpointArnT&& value) {
    SetTargetEndpointArn(std::forward<TargetEndpointArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_replicationConfigArn;

  Aws::String m_replicationConfigIdentifier;

  MigrationTypeValue m_replicationType{MigrationTypeValue::NOT_SET};

  Aws::String m_tableMappings;

  Aws::String m_replicationSettings;

  Aws::String m_supplementalSettings;

  ComputeConfig m_computeConfig;

  Aws::String m_sourceEndpointArn;

  Aws::String m_targetEndpointArn;
  bool m_replicationConfigArnHasBeenSet = false;
  bool m_replicationConfigIdentifierHasBeenSet = false;
  bool m_replicationTypeHasBeenSet = false;
  bool m_tableMappingsHasBeenSet = false;
  bool m_replicationSettingsHasBeenSet = false;
  bool m_supplementalSettingsHasBeenSet = false;
  bool m_computeConfigHasBeenSet = false;
  bool m_sourceEndpointArnHasBeenSet = false;
  bool m_targetEndpointArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace DatabaseMigrationService
}  // namespace Aws
