﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>
#include <aws/resiliencehub/model/EventType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ResilienceHub {
namespace Model {

/**
 * <p>Indicates an event you would like to subscribe and get notification for.
 * Currently, Resilience Hub supports notifications only for <b>Drift detected</b>
 * and <b>Scheduled assessment failure</b> events.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/resiliencehub-2020-04-30/EventSubscription">AWS
 * API Reference</a></p>
 */
class EventSubscription {
 public:
  AWS_RESILIENCEHUB_API EventSubscription() = default;
  AWS_RESILIENCEHUB_API EventSubscription(Aws::Utils::Json::JsonView jsonValue);
  AWS_RESILIENCEHUB_API EventSubscription& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_RESILIENCEHUB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The type of event you would like to subscribe and get notification for.
   * Currently, Resilience Hub supports notifications only for <b>Drift detected</b>
   * (<code>DriftDetected</code>) and <b>Scheduled assessment failure</b>
   * (<code>ScheduledAssessmentFailure</code>) events.</p>
   */
  inline EventType GetEventType() const { return m_eventType; }
  inline bool EventTypeHasBeenSet() const { return m_eventTypeHasBeenSet; }
  inline void SetEventType(EventType value) {
    m_eventTypeHasBeenSet = true;
    m_eventType = value;
  }
  inline EventSubscription& WithEventType(EventType value) {
    SetEventType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique name to identify an event subscription.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  EventSubscription& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the Amazon Simple Notification Service topic.
   * The format for this ARN is:
   * <code>arn:partition:sns:region:account:topic-name</code>. For more information
   * about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetSnsTopicArn() const { return m_snsTopicArn; }
  inline bool SnsTopicArnHasBeenSet() const { return m_snsTopicArnHasBeenSet; }
  template <typename SnsTopicArnT = Aws::String>
  void SetSnsTopicArn(SnsTopicArnT&& value) {
    m_snsTopicArnHasBeenSet = true;
    m_snsTopicArn = std::forward<SnsTopicArnT>(value);
  }
  template <typename SnsTopicArnT = Aws::String>
  EventSubscription& WithSnsTopicArn(SnsTopicArnT&& value) {
    SetSnsTopicArn(std::forward<SnsTopicArnT>(value));
    return *this;
  }
  ///@}
 private:
  EventType m_eventType{EventType::NOT_SET};

  Aws::String m_name;

  Aws::String m_snsTopicArn;
  bool m_eventTypeHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_snsTopicArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
