// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package cmd

import (
	"context"
	"fmt"
	"testing"

	filesystem_options "code.forgejo.org/f3/gof3/v3/forges/filesystem/options"
	forgejo_options "code.forgejo.org/f3/gof3/v3/forges/forgejo/options"
	"code.forgejo.org/f3/gof3/v3/options"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	f3_tests "code.forgejo.org/f3/gof3/v3/tree/tests/f3"
	tests_forge "code.forgejo.org/f3/gof3/v3/tree/tests/f3/forge"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func Test_CmdMirrorArguments(t *testing.T) {
	ctx := context.Background()

	output, err := runApp(ctx, "f3", "mirror", "--from-type", "garbage")
	assert.ErrorContains(t, err, `allowed values are `)
	assert.Contains(t, output, "Incorrect Usage:")
}

func Test_CmdMirrorIntegrationDefaultToPath(t *testing.T) {
	ctx := context.Background()

	fixtureOptions := tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t)
	fixtureTree := generic.GetFactory("f3")(ctx, fixtureOptions)
	log := fixtureTree.GetLogger()
	log.Trace("======= build fixture")
	f3_tests.TreeBuild(t, "CmdMirrorDefault", fixtureOptions, fixtureTree)

	log.Trace("======= create mirror")
	mirrorOptions := tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t)
	mirrorTree := generic.GetFactory("f3")(ctx, mirrorOptions)

	p := "/forge/users/10111"
	log.Trace("======= mirror %s", p)
	output, err := runApp(ctx, "f3", "--verbose", "mirror",
		"--from-filesystem-directory", fixtureOptions.(options.URLInterface).GetURL(),
		"--from-path", p,
		"--to-filesystem-directory", mirrorOptions.(options.URLInterface).GetURL(),
	)
	log.Trace("======= assert")
	assert.NoError(t, err)
	require.Contains(t, output, fmt.Sprintf("mirror %s (filesystem", p))

	mirrorTree.WalkAndGet(ctx, generic.NewWalkOptions(nil))
	found := mirrorTree.Find(generic.NewPathFromString(p))
	require.NotEqualValues(t, found, generic.NilNode)
	assert.EqualValues(t, p, found.GetCurrentPath().String())
}

func Test_CmdMirrorIntegrationSpecificToPath(t *testing.T) {
	ctx := context.Background()

	mirrorOptions := tests_forge.GetFactory(forgejo_options.Name)().NewOptions(t)
	mirrorTree := generic.GetFactory("f3")(ctx, mirrorOptions)

	fixtureOptions := tests_forge.GetFactory(filesystem_options.Name)().NewOptions(t)
	fixtureTree := generic.GetFactory("f3")(ctx, fixtureOptions)

	log := fixtureTree.GetLogger()
	creator := f3_tests.NewCreator(t, "CmdMirrorSpecific", log)

	log.Trace("======= build fixture")

	var fromPath string
	{
		fixtureUserID := "userID01"
		fixtureProjectID := "projectID01"

		userFormat := creator.GenerateUser()
		userFormat.SetID(fixtureUserID)
		users := fixtureTree.MustFind(generic.NewPathFromString("/forge/users"))
		user := users.CreateChild(ctx)
		user.FromFormat(userFormat)
		user.Upsert(ctx)
		require.EqualValues(t, user.GetID(), users.GetIDFromName(ctx, userFormat.UserName))

		projectFormat := creator.GenerateProject()
		projectFormat.SetID(fixtureProjectID)
		projects := user.MustFind(generic.NewPathFromString("projects"))
		project := projects.CreateChild(ctx)
		project.FromFormat(projectFormat)
		project.Upsert(ctx)
		require.EqualValues(t, project.GetID(), projects.GetIDFromName(ctx, projectFormat.Name))

		fromPath = fmt.Sprintf("/forge/users/%s/projects/%s", userFormat.UserName, projectFormat.Name)
	}

	log.Trace("======= create mirror")

	var toPath string
	var projects generic.NodeInterface
	{
		userFormat := creator.GenerateUser()
		users := mirrorTree.MustFind(generic.NewPathFromString("/forge/users"))
		user := users.CreateChild(ctx)
		user.FromFormat(userFormat)
		user.Upsert(ctx)
		require.EqualValues(t, user.GetID(), users.GetIDFromName(ctx, userFormat.UserName))

		projectFormat := creator.GenerateProject()
		projects = user.MustFind(generic.NewPathFromString("projects"))
		project := projects.CreateChild(ctx)
		project.FromFormat(projectFormat)
		project.Upsert(ctx)
		require.EqualValues(t, project.GetID(), projects.GetIDFromName(ctx, projectFormat.Name))

		toPath = fmt.Sprintf("/forge/users/%s/projects/%s", userFormat.UserName, projectFormat.Name)
	}

	log.Trace("======= mirror %s", fromPath)
	output, err := runApp(ctx, "f3", "--verbose", "mirror",
		"--from-type", filesystem_options.Name,
		"--from-path", fromPath,
		"--from-filesystem-directory", fixtureOptions.(options.URLInterface).GetURL(),

		"--to-type", forgejo_options.Name,
		"--to-path", toPath,
		"--to-forgejo-user", mirrorOptions.(options.AuthInterface).GetUsername(),
		"--to-forgejo-password", mirrorOptions.(options.AuthInterface).GetPassword(),
		"--to-forgejo-url", mirrorOptions.(options.URLInterface).GetURL(),
	)
	assert.NoError(t, err)
	log.Trace("======= assert")
	require.Contains(t, output, fmt.Sprintf("mirror %s", fromPath))
	projects.List(ctx)
	require.NotEmpty(t, projects.GetChildren())
	log.Trace("======= project %s", projects.GetChildren()[0])
}
