// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/TriggerUA5.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// UA5 \f$ \eta \f$ distributions at 200 and 900 GeV
  class UA5_1986_I233599 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(UA5_1986_I233599);


    /// @name Analysis methods
    /// @{

    /// Set up projections and histograms
    void init() {
      declare(TriggerUA5(), "Trigger");
      declare(Beam(), "Beams");
      declare(ChargedFinalState(Cuts::abseta < 5.0), "CFS50");

      for (double eVal : allowedEnergies()) {
        int en(eVal);
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_c[en+10], "_sumWtrigNSD"+toString(en));
        book(_c[en+20], "_sumWtrig"+toString(en));
        unsigned int offset = 2*int(en == 900);
        book(_h[en+10], 1, 1, 1+offset);
        book(_h[en+20], 1, 1, 2+offset);
        for (size_t i = 0; i < (en==900? 9 : 6); ++i) {
          book(_c[en+i], "TMP/sumWn"+toString(en+i));
          book(_h[en+i], 2+int(en==900), 1, i+1);
        }
      }
      raiseBeamErrorIf(_sqs < 0);
    }


    /// Fill eta histograms (in Nch bins)
    void analyze(const Event& event) {
      // Trigger
      const TriggerUA5& trigger = apply<TriggerUA5>(event, "Trigger");
      if (!trigger.sdDecision()) vetoEvent;
      const bool isNSD = trigger.nsdDecision();

      // Get the index corresponding to the max Nch range histo/sum(w) vector index
      const ChargedFinalState& cfs50 = apply<ChargedFinalState>(event, "CFS50");
      const int numP = cfs50.size();
      const int ni = (int)floor(static_cast<float>(numP-2)/10.0);
      const int num_idx = min(ni, (_sqs==900? 8 : 5));

      // Update weights
      _c[_sqs+20]->fill();
      if (isNSD) {
        _c[_sqs+10]->fill();
        if (num_idx >= 0) _c[_sqs+num_idx]->fill();
      }

      // Fill histos
      for (const Particle& p : cfs50.particles()) {
        const double eta = p.abseta();
        _h[_sqs+20]->fill(eta);
        if (!isNSD)  continue;
        _h[_sqs+10]->fill(eta);
        if (num_idx >= 0) _h[_sqs+num_idx]->fill(eta);
      }
    }


    /// Scale histos
    void finalize() {
      for (const auto& item : _c) {
        const double w = item.second->val();
        if (!isZero(w)) scale(_h[item.first], 0.5/w);
      }
    }


  private:

    /// @name Weight counters
    /// @{
    map<int,CounterPtr> _c;
    /// @}

    /// @name Histograms
    /// @{
    map<int,Histo1DPtr> _h;
    /// @}
    int _sqs = -1;

  };

  RIVET_DECLARE_ALIASED_PLUGIN(UA5_1986_I233599, UA5_1986_S1583476);
}
