------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ A T T R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.35 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Exp_Ch9;  use Exp_Ch9;
with Itypes;   use Itypes;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Res;  use Sem_Res;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Exp_Attr is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Expand_Attribute_Callable          (N : Node_Id);
   procedure Expand_Attribute_Count             (N : Node_Id);
   procedure Expand_Attribute_Image             (N : Node_Id);
   procedure Expand_Attribute_Size              (N : Node_Id);
   procedure Expand_Attribute_Storage_Size      (N : Node_Id);
   procedure Expand_Attribute_Terminated        (N : Node_Id);

   -------------------------------
   -- Expand_Attribute_Callable --
   -------------------------------

   --  Transforms 'Callable attribute into a call to the Callable function.

   procedure Expand_Attribute_Callable (N : Node_Id) is
   begin
      Rewrite_Substitute_Tree (N,
        Build_Call_With_Task (Prefix (N), RTE (RE_Callable)));
      Analyze (N);
      Resolve (N, Standard_Boolean);
   end Expand_Attribute_Callable;

   ----------------------------
   -- Expand_Attribute_Count --
   ----------------------------

   --  Transforms 'Count attribute into a call to the Count function

   procedure Expand_Attribute_Count (N : Node_Id) is
      Loc    : constant Source_Ptr := Sloc (N);
      Typ    : constant Entity_Id  := Etype (N);
      Entnam : Node_Id;
      Index  : Node_Id;
      Call   : Node_Id;

   begin
      if Nkind (Prefix (N)) = N_Indexed_Component then
         Entnam := Prefix (Prefix (N));
         Index := First (Expressions (Prefix (N)));
      else
         Entnam := Prefix (N);
         Index := Empty;
      end if;

      Call :=
        Make_Function_Call (Loc,
          Name => New_Reference_To (RTE (RE_Task_Count), Loc),
          Parameter_Associations => New_List (
            Entry_Index_Expression (Entnam, Index)));

      --  The call returns type Natural, but the context is universal integer,
      --  so any integer type is allowed. The attribute was already resolved,
      --  so its Etype is the required result type. If the base type of the
      --  context type is other than Standard.Integer, we put in a conversion
      --  to the required type. This can be a normal typed conversion, since
      --  both the input and output types of the conversion are integer types.

      if Base_Type (Typ) /= Standard_Integer then
         Rewrite_Substitute_Tree (N,
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Expression => Call));
      else
         Rewrite_Substitute_Tree (N, Call);
      end if;

      Analyze (N);
      Resolve (N, Typ);

   end Expand_Attribute_Count;

   ----------------------------
   -- Expand_Attribute_Image --
   ----------------------------

   --  For Boolean, Character, and numeric types, typ'Image (X) expands to:

   --    [B : string (1 .. max);
   --     B (1 .. System.Image.Image_styp (styp (X), B'Address)) ]

   --  where styp is one of the standard supported types. For user defined
   --  enumeration types, the transformation is to:

   --    Table (Enum'Pos (X)).all

   --  where table is the special table declared in the front end and
   --  constructed by special code in Gigi.

   procedure Expand_Attribute_Image (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);

      Typ : constant Entity_Id := Entity (Prefix (N));
      --  Prefix type of image attribute

      Btyp : constant Entity_Id := Base_Type (Typ);
      --  The base type of the prefix type

      Maxb : Nat;
      --  Maximum length required for buffer (including trailing NUL) used
      --  in call to function in System.Image package.

      Ient : Entity_Id;
      --  Entity for routine to call in Image package

      Ityp : Entity_Id := Btyp;
      --  Type used for first argument to image routine (styp above).
      --  Usually, but not always the same as the base type.

      B : Entity_Id;
      --  The entity used for the buffer in the expansion

   begin
      if Btyp = Standard_Boolean then
         Maxb := 6;
         Ient := RTE (RE_Img_B);

      --  In the case of Integer, we don't bother to have separate routines
      --  for all integer lengths, it's good enough to just handle the Integer
      --  and Long_Long_Integer cases, and use the smaller of these that fits.

      --  What about modular types ???

      elsif Is_Integer_Type (Btyp) then
         if Esize (Btyp) <= Esize (Standard_Integer) then
            Ityp := Standard_Integer;
            Maxb := Standard_Integer_Width + 1;
            Ient := RTE (RE_Img_I);
         else
            Ityp := Standard_Long_Long_Integer;
            Maxb := Standard_Long_Long_Integer_Width + 1;
            Ient := RTE (RE_Img_LLI);
         end if;

      elsif Btyp = Standard_Short_Float then
         Maxb := Standard_Short_Float_Digits + 10;
         Ient := RTE (RE_Img_SF);

      elsif Btyp = Standard_Float then
         Maxb := Standard_Float_Digits + 10;
         Ient := RTE (RE_Img_F);

      elsif Btyp = Standard_Long_Float then
         Maxb := Standard_Long_Float_Digits + 10;
         Ient := RTE (RE_Img_LF);

      elsif Btyp = Standard_Long_Long_Float then
         Maxb := Standard_Long_Float_Digits + 10;
         Ient := RTE (RE_Img_LLF);

      elsif Btyp = Standard_Character then
         Maxb := 10;
         Ient := RTE (RE_Img_C);

      elsif Btyp = Standard_Wide_Character then
         Maxb := 10;
         Ient := RTE (RE_Img_WC);

      --  Only other possibility that is currently supported is the user
      --  defined enumeration type case (other unsupported cases, notably
      --  the real cases, should have been caught as errors earlier on)

      else
         Rewrite_Substitute_Tree (N,
           Make_Explicit_Dereference (Loc,
             Prefix =>
               Make_Indexed_Component (Loc,
                 Prefix =>
                   New_Reference_To
                     (Lit_Name_Table (Entity (Prefix (N))), Loc),
                 Expressions => New_List (
                   Make_Attribute_Reference (Loc,
                     Prefix         => Prefix (N),
                     Attribute_Name => Name_Pos,
                     Expressions    => New_List (First (Expressions (N))))))));

         Analyze (N);
         Resolve (N, Standard_String);
         return;
      end if;

      --  If we fall through, we have one of the cases that is handled by
      --  calling one of the routines in the System.Image package

      B := Make_Defining_Identifier (Loc, New_Internal_Name ('B'));

      Rewrite_Substitute_Tree (N,
        Make_Expression_Actions (Loc,
           Actions => New_List (
             Make_Object_Declaration (Loc,
               Defining_Identifier => B,
               Object_Definition   =>
                 Make_Subtype_Indication (Loc,
                   Subtype_Mark => New_Reference_To (Standard_String, Loc),
                   Constraint   =>
                     Make_Index_Or_Discriminant_Constraint (Loc,
                       Constraints => New_List (
                         Make_Range (Loc,
                           Low_Bound  => Make_Integer_Literal (Loc, Uint_1),
                           High_Bound =>
                             Make_Integer_Literal (Loc,
                               Intval => UI_From_Int (Maxb)))))))),
           Expression =>
             Make_Slice (Loc,
               Prefix => New_Reference_To (B, Loc),
               Discrete_Range =>
                 Make_Range (Loc,
                   Low_Bound  => Make_Integer_Literal (Loc, Uint_1),
                   High_Bound =>
                     Make_Function_Call (Loc,
                       Name => New_Reference_To (Ient, Loc),
                       Parameter_Associations => New_List (
                         Make_Type_Conversion (Loc,
                           Subtype_Mark => New_Reference_To (Ityp, Loc),
                           Expression => New_Copy (First (Expressions (N)))),
                         Make_Attribute_Reference (Loc,
                           Prefix          => New_Reference_To (B, Loc),
                           Attribute_Name  => Name_Address)))))));

      Analyze (N);
      Resolve (N, Standard_String);
   end Expand_Attribute_Image;

   ---------------------------
   -- Expand_Attribute_Size --
   ---------------------------

   --  Transforms X'Size into a call to the first dispatching operation
   --  contained  in the Dispatch Table pointed by X._tag. This first operation
   --  happens to be  the implicit _Size function giving the size of a tagged
   --  object. We can't just expand a call to this function, and rely on
   --  further expansion to transform it into a dispatch call, because _size
   --  may not be visible at this point.

   procedure Expand_Attribute_Size (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Pref        : constant Node_Id    := Prefix (N);
      Typ         : constant Entity_Id  := Etype (Pref);
      Context_Typ : constant Entity_Id := Etype (N);
      New_Node    : Node_Id;

   begin
      if Is_Class_Wide_Type (Typ) then

         New_Node :=
           Make_Indexed_Component (Loc,
             Prefix =>
               Make_Selected_Component (Loc,
                 Prefix => Make_DT_Access (Loc, Pref, Etype (Typ)),
                 Selector_Name => Make_DT_Component (Loc, Etype (Typ), 3)),
             Expressions => New_List (New_Copy (Pref)));

         if Context_Typ /= Universal_Integer then

            New_Node :=
               Make_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (Context_Typ, Loc),
                 Expression => New_Node);
         end if;

         Rewrite_Substitute_Tree (N, New_Node);
         Analyze (N);
         Resolve (N, Context_Typ);
      end if;
   end Expand_Attribute_Size;

   -----------------------------------
   -- Expand_Attribute_Storage_Size --
   -----------------------------------

   --  The case of access types is currently unimplemented (and should have
   --  been caught prior to this point, so we simply ignore this case here)

   --  The case of a task type results in the attribute reference being
   --  replaced by the literal zero, which indicates that it is not in
   --  general sensible to apply Storage_Size to a task type, since the
   --  storage size may depend on a dynamic expression, or on discriminants.

   --  For the case of a task object, if there is no pragma Storage_Size,
   --  then we also return the literal zero, otherwise if there is a
   --  Storage_Size pragma, then we replace the attribute reference by
   --  the expression:

   --    Universal_Integer (taskV!(name)._Size)

   --  to obtain the Size field of the record object associated with the task.

   procedure Expand_Attribute_Storage_Size (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Pref        : constant Node_Id    := Prefix (N);
      Typ         : constant Entity_Id  := Etype (Pref);

   begin
      if Is_Access_Type (Typ) then
         null;

      --  Task cases

      else
         pragma Assert (Is_Task_Type (Typ));

         --  Case of task type

         if (Nkind (Pref) in N_Entity_Name
               or else (Nkind (Pref) = N_Attribute_Reference
                         and then Present (Entity (Pref))))
              and then Is_Task_Type (Entity (Pref))
         then
            Replace_Substitute_Tree (N,
              Make_Integer_Literal (Loc, Uint_0));

         --  Case of task object

         else
            declare
               Rtyp : constant Entity_Id := Corresponding_Record_Type (Typ);

            begin
               --  Task object which has Storage_Size pragma

               w ("rtyp = ", Int (Rtyp));
               w ("le = ", Int (Last_Entity (Rtyp)));
               w ("ch = ", Int (Chars (Last_Entity (Rtyp))));

               if Chars (Last_Entity (Rtyp)) = Name_uSize then

                  Replace_Substitute_Tree (N,
                    Make_Type_Conversion (Loc,
                      Subtype_Mark =>
                        New_Reference_To (Universal_Integer, Loc),
                      Expression =>
                        Make_Selected_Component (Loc,
                          Prefix =>
                            Make_Unchecked_Type_Conversion (Loc,
                              Subtype_Mark =>
                                New_Reference_To
                                  (Corresponding_Record_Type (Typ), Loc),
                              Expression => New_Copy_Tree (Pref)),
                          Selector_Name =>
                            Make_Identifier (Loc, Name_uSize))));

               --  Task object not having Storage_Size pragma

               else
                  Replace_Substitute_Tree (N,
                    Make_Integer_Literal (Loc, Uint_0));
               end if;
            end;
         end if;

         Analyze (N);
         Resolve (N, Universal_Integer);
      end if;

   end Expand_Attribute_Storage_Size;

   ---------------------------------
   -- Expand_Attribute_Terminated --
   ---------------------------------

   --  Transforms 'Terminated attribute into a call to the Terminated function.

   procedure Expand_Attribute_Terminated (N : Node_Id) is
   begin
      Rewrite_Substitute_Tree (N,
        Build_Call_With_Task (Prefix (N), RTE (RE_Terminated)));
      Analyze (N);
      Resolve (N, Standard_Boolean);
   end Expand_Attribute_Terminated;

   ----------------------------------
   -- Expand_N_Attribute_Reference --
   ----------------------------------

   procedure Expand_N_Attribute_Reference (N : Node_Id) is
   begin
      case Get_Attribute_Id (Attribute_Name (N)) is

         --  Attributes requiring special expander action

         when Attribute_Callable =>
            Expand_Attribute_Callable (N);

         when Attribute_Count =>
            Expand_Attribute_Count (N);

         when Attribute_Image =>
            Expand_Attribute_Image (N);

         when Attribute_Size =>
            Expand_Attribute_Size (N);

         when Attribute_Storage_Size =>
            Expand_Attribute_Storage_Size (N);

         when Attribute_Terminated =>
            Expand_Attribute_Terminated (N);

         --  All other attributes need no expander action

         when others => null;
      end case;

   end Expand_N_Attribute_Reference;

end Exp_Attr;
