/*
 * dirlist.c
 *
 * This file is part of "locatedir"
 * by Alexander Kourakos <awk@vt.edu>
 *
 * Copyright (C) 1994 Alexander Kourakos
 * All rights reserved.
 *
 * This software may be freely copied, modified, and redistributed
 * provided that this copyright notice remains unchanged on all copies.
 *
 * You may not distribute this software, in whole or in part, as part of
 * any commercial product without the express consent of the author.
 *
 * There is no warranty or other guarantee of this software fitness
 * for any purpose. It is provided solely "as is".
 */

#include "locatedir.h"
#include <ctype.h>
#include <sys/stat.h>
#include <time.h>

/*
 * Attempts to open the directory list file. If it can't
 * be opened, creates it first.
 */

FILE   *
open_dirlistfile (const char *path)
{
  FILE   *dirlistfilep;
  struct stat statbuf;

/*
 * See if the dir list file is there.
 */
  if (stat (path, &statbuf))
  {
    if (errno == ENOENT)
    {
      /*
       * If it's not there, make it.
       */
      if (!quiet)
	fprintf (stderr, "Directory list file \"%s\" does not exist!\n", path);

      strcpy (startpath, homepath);
      build_dirlistfile (path);

      if ((dirlistfilep = fopen (path, "r")) == (FILE *) NULL)
	error ("unable to open directory list file");
    }
    else
      error ("couldn't stat directory list file");
  }
  else
  {
/*
 * If it is there, open it and get the starting directory.
 */
    if ((dirlistfilep = fopen (path, "r")) == (FILE *) NULL)
      error ("unable to open directory list file");

    if (fgets (startpath, MAX_PATH_LENGTH, dirlistfilep) == (char *) NULL)
      error ("error reading from directory list file");
    TRUNC_LAST_CHAR (startpath);
/*
 * If the file is really old, and we are allowing it, rebuild the file.
 */
    if (rebuild_if_old && ((time (NULL) - statbuf.st_mtime) > MAX_SECS))
    {
      if (!quiet)
	fprintf (stderr, "Directory list file is older than %d day%s!\n", MAX_DAYS, (MAX_DAYS == 1) ? "" : "s");

      if (fclose (dirlistfilep) == EOF)
	error ("unable to close directory list file");

      (void) remove (path);

      build_dirlistfile (path);

      if ((dirlistfilep = fopen (path, "r")) == (FILE *) NULL)
	error ("unable to open directory list file");
    }
  }

  return dirlistfilep;
}

/*
 * Creates the directory list file, and leaves it closed.
 */

void
build_dirlistfile (const char *pathname)
{
  char    line[MAX_PATH_LENGTH];
  char    command[MAX_PATH_LENGTH];
  FILE   *tmpfile1, *tmpfile2;
  char    tmpfile1name[MAX_NAME_LENGTH], tmpfile2name[MAX_NAME_LENGTH];
  FILE   *dirlistfile;

  fprintf (stderr, "Building directory tree list from \"%s\"...\n\n", startpath);

  /*
   * Yuck. I'd rather use one temporary file, but this will do.
   */
  tmpnam (tmpfile1name);
  tmpnam (tmpfile2name);

/*
 * Search the start path for directories.
 */
  sprintf (command, FIND_COMMAND " 2>/dev/null >%s", startpath, tmpfile1name);
  system (command);

  if (((tmpfile1 = fopen (tmpfile1name, "r")) == (FILE *) NULL) ||
      ((tmpfile2 = fopen (tmpfile2name, "w")) == (FILE *) NULL))
    error ("error opening temporary files");

/*
 * Loop through the output of find and mangle each line.
 */

  for (;;)
  {
    if (fgets (line, MAX_PATH_LENGTH, tmpfile1) == (char *) NULL)
      if (!feof (tmpfile1))
	error ("error reading from first temporary file");
      else
	break;

    mangle (line);

    if (fputs (line, tmpfile2) == EOF)
      error ("error writing to second temporary file");
  }

  if ((fclose (tmpfile1) == EOF) || (fclose (tmpfile2) == EOF))
    error ("error closing temporary files");

  if ((dirlistfile = fopen (pathname, "w")) == (FILE *) NULL)
    error ("unable to open directory list file");

  fputs (startpath, dirlistfile);
  fputc ('\n', dirlistfile);

  if (fclose (dirlistfile) == EOF)
    error ("error closing directory list file");

/*
 * Sort the file list.
 */

  sprintf (command, SORT_COMMAND " 2>/dev/null >> %s", tmpfile2name, pathname);
  system (command);

  (void) remove (tmpfile1name);
  (void) remove (tmpfile2name);
}

/*
 * Compares a directory name to a mangled path.
 * Returns:  0 if they don't match at all
 *          -1 if they match perfectly
 *          -2 if they match case insensitively
 *           n if they match (case insensitively) up to the nth character
 */

static int
compare (const char *name, const char *path)
{
  int     matches, case_matched;
  int     length = strlen (name);

  for (matches = 0, case_matched = 1; matches < length; matches++, name++, path++)
  {
    if ((*path == '\n') || (*path == DIR_SEP_CHAR))
      return matches;

    if (tolower (*name) != tolower (*path))
      return matches;

    if (*name != *path)
      case_matched = 0;
  }

  return
    ((*path == DIR_SEP_CHAR) || (*path == '\n')) ?
    (-1 - (case_matched == 0)) :
    matches;
}

/*
 * Attempts to find a good match to "dirname" in the given file, and
 * returns it in "foundpath".
 */

int
find_best_match (FILE * dirlistfile, const char *dirname, char *foundpath)
{
  char    line[MAX_PATH_LENGTH], mangled_cwd[MAX_PATH_LENGTH];
  int     watch_out_for_cwd, seen_match, seen_cwd;
  char    bestline[MAX_PATH_LENGTH], bestafterline[MAX_PATH_LENGTH];
  unsigned int best_rating, best_after_rating, last_rating, last_after_rating;
  unsigned int result;

  watch_out_for_cwd = 0;
  if (!strncmp (startpath, cwdpath, strlen (startpath)))
  {
    strcpy (mangled_cwd, cwdpath);
    strcat (mangled_cwd, "\n");
    mangle (mangled_cwd);
    if (compare (dirname, mangled_cwd) != 0)
      watch_out_for_cwd = 1;
  }

  best_rating = best_after_rating = last_rating = last_after_rating = 0;

  seen_match = seen_cwd = 0;
  for (;;)
  {
    if (fgets (line, MAX_PATH_LENGTH, dirlistfile) == (char *) NULL)
      if (feof (dirlistfile))
	break;
      else
	error ("error reading directory list file");

    if (watch_out_for_cwd)
    {
      if (!strcmp (mangled_cwd, line))
      {
	seen_cwd = 1;
	continue;
      }

      result = (unsigned int) compare (dirname, line);

      if (result != 0)
	seen_match = 1;
      else if (seen_match)
	break;
      else
	continue;

      if (seen_cwd)
      {
	if (result > best_after_rating)
	{
	  strcpy (bestafterline, line);
	  last_after_rating = best_after_rating;
	  best_after_rating = result;
	}
      }
      else
      {
	if (result > best_rating)
	{
	  strcpy (bestline, line);
	  last_rating = best_rating;
	  best_rating = result;
	}
      }

    }
    else
    {
      result = (unsigned int) compare (dirname, line);

      if (result != 0)
	seen_match = 1;
      else if (seen_match)
	break;
      else
	continue;

      if (result > best_rating)
      {
	strcpy (bestline, line);
	last_rating = best_rating;
	best_rating = result;
      }
    }
  }

  if ((best_after_rating >= best_rating) && (best_after_rating != 0))
  {
    best_rating = best_after_rating;
    strcpy (bestline, bestafterline);
    last_rating = last_after_rating;
  }

  if (best_rating == 0)
    return 1;

  strcpy (foundpath, bestline);
  demangle (foundpath);

  return 0;
}

/*
 * end of file 
 */
