use core:lang;
use lang:bs;
use lang:bs:macro;
use lang:bs:unsafe;

/**
 * A column declaration.
 */
class Column {
	// Name of the column.
	Str name;

	// Datatype.
	SQLType datatype;

	// Is this column a primary key?
	Bool primaryKey;

	// Is this column allowed to contain NULL?
	Bool allowNull;

	// Unique?
	Bool unique;

	// Auto-increment?
	Bool autoIncrement;

	// Default value?
	SQLLiteral? default;


	init(SStr name, SQLType type) {
		init {
			name = name.v;
			datatype = type;
		}
	}

	// Called from the syntax.
	void setPrimary(Str x) { primaryKey = true; }
	void setAllowNull(Str x) { allowNull = true; }
	void setUnique(Str x) { unique = true; }
	void setAutoIncrement(Str x) { autoIncrement = true; }
	void setDefault(SQLLiteral x) { default = x; }

	// Called from the syntax, gives a decent error message on how we're different from SQL.
	void setNotNull(SStr x) {
		throw SyntaxError(x.pos, "Columns are NOT NULL by default. Use ALLOW NULL to allow nullable columns.");
	}

	// Create an SQL part for this column.
	void toSQL(QueryStrBuilder to) {
		to.name(name);
		to.put(" ");
		to.type(datatype);
		modifiers(to);
	}

	void toS(StrBuf to) : override {
		QueryStrBuilder b;
		toSQL(b);
		to << b.build;
		if (primaryKey)
			to << " (primary key)";
	}

	// To schema.
	Schema:Column toSchema() {
		Schema:Attributes attrs;
		if (primaryKey)
			attrs += Schema:Attributes:primaryKey;
		if (!allowNull)
			attrs += Schema:Attributes:notNull;
		if (unique)
			attrs += Schema:Attributes:unique;
		if (autoIncrement)
			attrs += Schema:Attributes:autoIncrement;

		Schema:Column out(name, datatype.sqlType, attrs);
		if (default)
			out.default = default.toSQL();

		out;
	}

	// See if this column has some kind of default value. I.e. it is OK to leave one without a value?
	Bool hasDefault(Bool multiPK) {
		if (default)
			return true;
		if (primaryKey & !multiPK)
			return true;
		if (allowNull)
			return true;
		if (autoIncrement)
			return true;
		false;
	}

	private void modifiers(QueryStrBuilder to) {
		if (!allowNull) {
			to.put(" NOT NULL");
		}
		if (unique) {
			to.put(" UNIQUE");
		}
		if (autoIncrement) {
			to.put(" ");
			to.autoIncrement();
		}
		if (default) {
			to.put(" DEFAULT " + default.toSQL());
		}
	}
}


/**
 * Index for a table.
 */
class Index {
	// Name of the index (might be auto-generated).
	Str name;

	// Columns.
	Str[] columns;

	// Create.
	init(Str name, Str[] columns) {
		init {
			name = name;
			columns = columns;
		}
	}

	// To schema.
	Schema:Index toSchema()	{
		return Schema:Index(name, columns);
	}

	// To SQL statement.
	void toSQL(QueryStrBuilder to, Str table) {
		to.put("CREATE INDEX ");
		to.name(name);
		to.put(" ON ");
		to.name(table);
		to.put("(");
		for (i, c in columns) {
			if (i > 0)
				to.put(", ");
			to.name(c);
		}
		to.put(");");
	}

	// To string. No SQL escaping.
	void toS(StrBuf to) : override {
		to << "INDEX " << name << " ON ?(" << join(columns, ", ") << ")";
	}
}

/**
 * Declaration of an entire table.
 */
class Table {
	// Name of the table.
	Str name;

	// Columns in the table.
	Column[] columns;

	// Indices for this table.
	Index[] indices;

	init(SStr name) {
		init {
			name = name.v;
		}
	}

	// Find a column.
	Column? find(Str name) {
		for (c in columns)
			if (c.name == name)
				return c;
		null;
	}

	// Add column (called from syntax).
	void add(Column col) {
		columns << col;
	}

	// Add primary key(s) (called from syntax).
	void add(Array<SStr> cols) {
		for (c in columns)
			if (c.primaryKey)
				throw SyntaxError(cols[0].pos, "Only one instance of the PRIMARY KEY keyword may be present for each table.");

		for (c in cols) {
			unless (col = find(c.v))
				throw SyntaxError(c.pos, "No column named ${c.v} was declared in this table.");
			col.primaryKey = true;
		}
	}

	// Add index.
	void add(SrcPos pos, Index index) {
		for (i in indices)
			if (i.name == index.name)
				throw SyntaxError(pos, "The index ${index.name} already exists.");

		indices << index;
	}

	// Check if there are multiple primary keys.
	Bool multiplePK() {
		Nat count = 0;
		for (c in columns)
			if (c.primaryKey)
				count++;
		count > 1;
	}

	// Figure out what index, if any, has an implicit AUTOINCREMENT. Returns "count" if none has it.
	Nat implicitAutoIncrementColumn() {
		Nat count = columns.count();
		Nat firstPrimary = count();

		for (i, c in columns) {
			if (c.primaryKey) {
				if (firstPrimary < count)
					return count;
				firstPrimary = i;
			}
		}

		var col = columns[firstPrimary];
		// Reset if it is either already marked as autoincrement, or if it is of the improper type.
		if (col.autoIncrement) {
			firstPrimary = count;
		} else if (!col.datatype.sqlType.sameType(QueryType:integer)) {
			firstPrimary = count;
		}

		return firstPrimary;
	}

	// Create a Schema from this table declaration.
	Schema toSchema() {
		Schema:Column[] cols;
		for (c in columns)
			cols << c.toSchema();

		Schema:Index[] inds;
		for (i in indices)
			inds << i.toSchema();

		return Schema(name, cols, inds);
	}

	// Create an SQL statement for this table declaration.
	void toSQL(QueryStrBuilder to, Bool ifNotExists, Bool implicitAutoIncrement, Str[] options) {
		to.put("CREATE TABLE ");
		if (ifNotExists)
			to.put("IF NOT EXISTS ");
		to.name(name);
		to.put(" (");

		Str[] pk;
		for (col in columns)
			if (col.primaryKey)
				pk << col.name;

		Nat implicitCol = if (implicitAutoIncrement) { columns.count; } else { implicitAutoIncrementColumn(); };
		for (i, col in columns) {
			if (i > 0)
				to.put(", ");
			col.toSQL(to);

			// Note: Some databases (e.g. SQLite) require that the PRIMARY KEY is specified here in
			// order for AUTOINCREMENT to work properly.
			if (pk.count == 1 & col.primaryKey)
				to.put(" PRIMARY KEY");

			if (implicitCol == i) {
				to.put(" ");
				to.autoIncrement();
			}
		}

		if (pk.count > 1) {
			to.put(", PRIMARY KEY(");
			for (i, k in pk) {
				if (i > 0)
					to.put(", ");
				to.name(k);
			}
			to.put(")");
		}

		to.put(")");
		if (options.any) {
			to.put(" ");
			for (i, k in options) {
				if (i > 0)
					to.put(", ");
				to.put(k);
			}
		}
		to.put(";");
	}
}

/**
 * Helper type for creating indices.
 */
class IndexDecl {
	SrcPos pos;
	SStr table;
	Index index;

	init(SrcPos pos, SStr name, SStr table, SStr[] cols) {
		Str[] c;
		for (x in cols)
			c << x.v;

		init {
			pos = pos;
			table = table;
			index(name.v, c);
		}
	}

	init(SrcPos pos, SStr table, SStr[] cols) {
		StrBuf name;
		name << table.v;

		Str[] c;
		for (x in cols) {
			c << x.v;
			name << "_" << x.v;
		}

		init {
			pos = pos;
			table = table;
			index(name.toS, c);
		}
	}
}

/**
 * Database description.
 */
class Database {
	// Tables declared (indices are stored inside each table).
	Table[] tables;

	// Add a table.
	void add(Table decl) {
		tables.push(decl);
	}

	// Add an index.
	void add(IndexDecl decl) {
		unless (table = find(decl.table.v))
			throw SyntaxError(decl.table.pos, "The table ${decl.table.v} was not declared (yet).");

		table.add(decl.pos, decl.index);
	}

	// Find a table.
	Table? find(Str name) {
		// TODO: Speedier lookup?
		for (table in tables)
			if (table.name == name)
				return table;
		null;
	}
}

/**
 * Declaration of a database.
 */
class DatabaseDecl extends NamedDecl {
	SStr name;
	Scope scope;
	Database contents;

	init(SStr name, Scope scope, Database contents) {
		init() {
			name = name;
			scope = scope;
			contents = contents;
		}
	}

	Named doCreate() {
		DatabaseType(name, contents);
	}
}

/**
 * Type stored in the name tree.
 */
class DatabaseType extends Type {
	// Contents of the database.
	Database contents;

	// All queries that operate on this database. This allows us to prepare all statements up-front,
	// which avoids surprises for applications down the line, both in terms of performance and in
	// the case some query results in an error due to incorrect database contents.
	CachedQuery[] queries;

	// Create.
	init(SStr name, Database contents) {
		init(name.v, TypeFlags:typeClass) {
			contents = contents;
		}

		setSuper(named{TypedConnection});
		addCtors();
	}

	private void addCtors() {
		{
			BSTreeCtor ctor([thisParam(this), ValParam(named{DBConnection}, "db")], SrcPos());
			CtorBody body(ctor, Scope());

			Actuals params;
			params.add(LocalVarAccess(SrcPos(), body.parameters[1]));
			params.add(lang:bs:util:TObjectLiteral(SrcPos(), this));
			body.add(InitBlock(SrcPos(), body, params));

			ctor.body = body;
			add(ctor);
		}
		{
			BSTreeCtor ctor([
								thisParam(this),
								ValParam(named{DBConnection}, "db"),
								ValParam(named{MigrationPolicy}, "policy")
							], SrcPos());
			CtorBody body(ctor, Scope());

			Actuals params;
			params.add(LocalVarAccess(SrcPos(), body.parameters[1]));
			params.add(lang:bs:util:TObjectLiteral(SrcPos(), this));
			params.add(LocalVarAccess(SrcPos(), body.parameters[2]));
			body.add(InitBlock(SrcPos(), body, params));

			ctor.body = body;
			add(ctor);
		}
	}
}


/**
 * Base class inherited from the DBType class.
 */
class TypedConnection {
	// Underlying database connection.
	DBConnection connection;

	// Create and verify the database structure.
	init(DBConnection conn, DatabaseType t) {
		self(conn, t, MigrationPolicy:default);
	}

	// Create and verify the database structure, also supply a migration policy.
	init(DBConnection conn, DatabaseType t, MigrationPolicy policy) {
		init { connection = conn; }

		// Make sure the database is in the state we expect it to be.
		verifyDatabaseSchema(conn, t.contents, policy);

		// Prepare statements now to avoid performance problems and to detect errors early. Note
		// that we use 'as thread' to avoid the implicit copy here (we do the same for the objects
		// we embed in the generated machine code, so this is not too bad).
		var queries = as thread Compiler { t.queries; };
		for (q in t.queries) {
			conn.prepare(q);
		}
	}

	// Close.
	void close() {
		connection.close();
	}

	// Get a cached prepared statement based on its ID.
	Statement prepare(CachedQuery cached) {
		connection.prepare(cached);
	}
}
